// Utilities to estimate the MA model of the masking threshold, block by block

function [A,Sigma2] = modele_AR_InvMaskMPEG1(x,p,NFFT,Fs)
    
    // Gael Mahe', november 2013

    // estimate the MA model of the masking threshold 
    // according to the MPEG-1 psychoacoustical model, 
    // block by block

    // INPUTS :
    // x : signal
    // p : model order
    // NFFT : size of the blocks of computation
    // Fs : sampling frequency
    
    // OUTPUTS
    // considering H(z) = (1+A(z)) / sigma the MA model of a block,
    // A = sequence of the coefficients of A(z), frame by frame, 
    // Sigma2 = sequence of the values of sigma^2
    
    // 50%-overlapping blocks
    // Each AR-model will be used for the central NFFT/2 samples of a block

    // MPEG-1 psychoacoustical model
    load("MPEG_mpa1_"+string(round(Fs/1000))+"k.dat");
    exec("mpeg_mpa1.sci");
    // NFFT must be 512
    
    nb_blocs = ceil(length(x)/(NFFT/2));
    x = [zeros(1,NFFT/4) x];
    x = [x zeros(1,nb_blocs*NFFT/2+NFFT/2-length(x))];
        
    // Outputs
    A = zeros(nb_blocs,p);
    Sigma2 = zeros(nb_blocs,1);
    
    index = 1:NFFT;
    threshold0 = zeros(1,NFFT/2);
    
    for bloc=1:nb_blocs
        
        // threshold de masquage selon modele MPEG1
        threshold_prev = threshold0;
        [SMR, threshold_dB , threshold2 , Ix] = mpeg_mpa1(x(index)', LTq_i, LTq_k, Table_z, Fi, Fk, Larg_f);
        threshold0 = (10).^(threshold_dB(1:$)/10); 

        // CTRL pre-echo
        if bloc ~= 1 then
            threshold = min(threshold0,2*threshold_prev);    
        else
            threshold = threshold0;
        end

        Mx = [threshold threshold($) threshold(NFFT/2:-1:2)]; 
        
        // AR model of the inverse of the masking threshold
        r = real(ifft(ones(1,NFFT)./(Mx+%eps))) * sqrt(NFFT); 
        [a,sigma2,rc]=lev(r(1:p+1)); 
        
        A(bloc,:) = a';
        Sigma2(bloc) = sigma2;
        
        index = index + NFFT/2;

    end
    
endfunction


function [A,Sigma2,Sigma_x] = modele_AR_InvSpectre(x,p,NFFT)
    
    // Gael Mahe', november 2013

    // estimate the MA model of power spectrum density, block by block

    // INPUTS :
    // x : signal
    // p : model order
    // NFFT : size of the blocks of computation
    
    // OUTPUTS
    // Sigma_x = sequence of the values of standard deviation of x
    // considering H(z) = (1+A(z)) / sigma the MA model of a block,
    // A = sequence of the coefficients of A(z), frame by frame, 
    // Sigma2 = sequence of the values of sigma^2
    
    // 50%-overlapping blocks
    // Each AR-model will be used for the central NFFT/2 samples of a block

    nb_blocs = ceil(length(x)/(NFFT/2));
    x = [zeros(1,NFFT/4) x];
    x = [x zeros(1,nb_blocs*NFFT/2+NFFT/2-length(x))];
    
    // Outputs
    A = zeros(nb_blocs,p);
    Sigma2 = zeros(nb_blocs,1);
    Sigma_x = zeros(nb_blocs,1);
    
    index = 1:NFFT;
    
    for bloc=1:nb_blocs
        
        win = window("hm",NFFT);
        X2 = abs(fft(x(index).*win)).^2 / sum(win.^2) ;
        
        // AR model of the inverse of the spectrum
        r = real(ifft(ones(1,NFFT)./(X2+%eps))); 
        [a,sigma2,rc]=lev(r(1:p+1)); 
        
        A(bloc,:) = a';
        Sigma2(bloc) = sigma2;
        
        Sigma_x(bloc) = stdev(x(index));
       
        index = index + NFFT/2;

    end
    
endfunction
