// Gael Mahe', October 2017
// Smoothes discontinuities occuring at block transitions

function w_new = correc_discont(w,paramDisc)
    
    // Gael Mahe', january 2017

    // INPUTS
    //    w = signal with discontinuities
    //    paramDisc = discontinuities smoothing parameters

    // OUPUTS
    //    w_new = w where discontinuities were reduced
        
    N = length(w);
    
    // smooths w through thresholding the derivative at the block-transitions
    dw = [0, diff(w)]; 
    dw_smooth = thresholding_blocks(dw,paramDisc.blockSize,paramDisc.Wmed,paramDisc.Wsig);
    w_smooth = zeros(1,N);
    w_smooth(1) = w(1);
    for n=2:N
        w_smooth(n) = w_smooth(n-1) + dw_smooth(n);
    end
    
    // Extract and LP-filter the discontinuous component of w
    w_disc = w - w_smooth;
    Ldiv2 = (paramDisc.Lfilt-1)/2;
    h = window("tr",Ldiv2*2+1); h = h/sum(h);
    w_disc_LP = convol(h,[w_disc,w_disc($)*ones(1,Ldiv2)]);
    w_disc_LP = w_disc_LP(Ldiv2+(1:N));
    
    // Reconstruct w with smoothed transitions
    w_new = round(w_smooth + w_disc_LP);
    
        
endfunction



function y = thresholding_blocks(x,blockSize,Wmed,Wsig)
    // thresholds only at block transitions
    N = length(x);
    NbBlocks = ceil(N/blockSize);
    // median value of x on a Wmed-wide neighborhood
    WmedDiv2 = (Wmed-1)/2;
    X = zeros(Wmed,N-Wmed+1);
    for i=1:Wmed
        X(i,:) = x((1:N-Wmed+1)+Wmed-i);
    end
    x_med = zeros(1,N);
    x_med(WmedDiv2+1:N-WmedDiv2) = median(X,1);
    for n=1:WmedDiv2
        x_med(n) = median(x(1:n+WmedDiv2));
    end
    for n=N-WmedDiv2+1:N
        x_med(n) = median(x(n-WmedDiv2:$));
    end
    // std value of dev = x - x_med on a Wsig-wide neighborhood
    // computed only at transitions (first sample of each block)
    WsigDiv2 = (Wsig-1)/2;
    DevTrans = zeros(Wsig,NbBlocks-1);
    for i=1:Wsig
        indices = (blockSize+1:blockSize:N)-WsigDiv2+i-1;
        if indices($)<=N then
            DevTrans(i,:) = x(indices)-x_med(indices);
        else
            DevTrans(i,1:$-1) = x(indices(1:$-1))-x_med(indices(1:$-1));
            DevTrans(i,$) = DevTrans(i-1,$);
        end
    end
    sigmaTrans = stdev(DevTrans,1);
    // discontintuities detection and erasing
    indices = blockSize+1:blockSize:N;
    IdiscTrans = find(abs(x(indices)-x_med(indices))>2*sigmaTrans);
    Idisc = IdiscTrans*blockSize + 1;
    y = x;
    y(Idisc) = x_med(Idisc);    
endfunction


